# 機能設計書 20-埋め込み層（Embedding Layers）

## 概要

本ドキュメントは、TensorFlowにおける埋め込み層（Embedding Layer）機能に関する設計書である。正の整数インデックスから密なベクトル表現への変換を行う`tf.keras.layers.Embedding`の仕様を詳述する。

### 本機能の処理概要

埋め込み層機能は、離散的なカテゴリカルデータ（単語ID、アイテムID等）を連続的な密ベクトル空間にマッピングする。語彙サイズ x 埋め込み次元の重み行列を保持し、入力インデックスに対応する行を検索して返す。

**業務上の目的・背景**：自然言語処理（NLP）において、単語は離散的なトークンとして表現される。埋め込み層はこれらの離散表現を密なベクトルに変換し、意味的な類似性を持つ単語が近い距離に配置される学習可能な表現空間を提供する。Word2Vec、GloVe等の事前学習済み埋め込みの初期値としても利用される。

**機能の利用シーン**：NLPモデル（BERT、GPT等）の入力層、推薦システムのアイテム/ユーザ埋め込み、カテゴリカル特徴量のエンコーディング、系列モデル（LSTM/GRU）の入力層。

**主要な処理内容**：
1. 埋め込み重み行列の初期化と管理
2. 整数インデックスから密ベクトルへの検索（embedding_lookup）
3. マスキング機能（mask_zero=True時、パディング値0をマスク）
4. 正則化（embeddings_regularizer/activity_regularizer）
5. 制約（embeddings_constraint）
6. 混合精度対応（出力のキャスト）

**関連システム・外部連携**：embedding_ops（低レベル埋め込み検索）、RNNレイヤー（マスキングとの連携）。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はモデル内部で使用される変換処理であり、画面に直接関連しない |

## 機能種別

計算処理（テンソル変換） / データ構造管理

## 入力仕様

### 入力パラメータ（コンストラクタ）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| input_dim | int | Yes | 語彙サイズ（最大インデックス+1） | 正の整数 |
| output_dim | int | Yes | 埋め込みベクトルの次元 | 正の整数 |
| embeddings_initializer | initializer | No（デフォルト'uniform'） | 重み初期化方法 | 有効なinitializer |
| embeddings_regularizer | regularizer | No | 重み正則化 | 有効なregularizer |
| activity_regularizer | regularizer | No | 出力正則化 | 有効なregularizer |
| embeddings_constraint | constraint | No | 重み制約 | 有効なconstraint |
| mask_zero | bool | No（デフォルトFalse） | パディング値0のマスク有効化 | - |
| input_length | int/tuple | No | 入力シーケンス長（固定長の場合） | 正の整数 |

### 入力データ（callメソッド）

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| inputs | tf.Tensor (int) | Yes | 整数インデックステンソル (batch_size, input_length) | int32/int64に変換される |

### 入力データソース

トークナイザの出力、カテゴリカル特徴量のエンコード結果。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| output | tf.Tensor | 密ベクトルテンソル (batch_size, input_length, output_dim) |
| mask | tf.Tensor (bool) | マスクテンソル (batch_size, input_length)。mask_zero=True時のみ |

### 出力先

次層（通常はRNN、Transformer、Dense等）への入力テンソル。

## 処理フロー

### 処理シーケンス

```
1. Embedding.__init__
   └─ パラメータ検証（input_dim > 0, output_dim > 0）
   └─ 初期化器/正則化器/制約の取得
   └─ autocast = False 設定
2. Embedding.build
   └─ 埋め込み重み行列の作成（input_dim x output_dim）
3. Embedding.call(inputs)
   └─ 入力dtypeがint32/int64でなければint32にキャスト
   └─ embedding_ops.embedding_lookup_v2で検索
   └─ 混合精度の場合は出力をcompute_dtypeにキャスト
4. Embedding.compute_mask(inputs)
   └─ mask_zero=True: inputs != 0 のマスク生成
   └─ mask_zero=False: None返却
```

### フローチャート

```mermaid
flowchart TD
    A[整数入力テンソル] --> B{dtype == int32/int64?}
    B -->|No| C[int32にキャスト]
    B -->|Yes| D[そのまま]
    C --> E[embedding_lookup_v2]
    D --> E
    E --> F{混合精度?}
    F -->|Yes| G[compute_dtypeにキャスト]
    F -->|No| H[出力]
    G --> H
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 正整数制約 | input_dimとoutput_dimは共に正の整数 | ValueError |
| BR-02 | 先頭レイヤー制約 | Embeddingはモデルの最初のレイヤーとしてのみ使用可能 | Sequential API |
| BR-03 | autocast無効 | 浮動小数点入力の自動キャストを無効化（int32キャストの精度問題回避） | 常時 |
| BR-04 | TF1 dtype互換 | TF1ではdtype未指定時にfloatx()（通常float32）を明示設定 | v2_dtype_behavior未有効時 |
| BR-05 | mask_zero注意 | mask_zero=True時、index 0は語彙に使用不可（input_dim = 語彙サイズ + 1） | mask_zero=True |
| BR-06 | GPU配置デフォルト | デフォルトでGPU上に配置（一部のオプティマイザはスパースGPUカーネル未対応） | GPU利用時 |

### 計算ロジック

- 埋め込み検索: `output = embeddings_matrix[inputs]`（各インデックスに対応する行を返す）
- マスク計算: `mask = inputs != 0`（mask_zero=True時）
- 出力形状: `(batch_size, input_length, output_dim)`

## データベース操作仕様

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ValueError | 非正パラメータ | input_dim <= 0 または output_dim <= 0 | 正の整数を指定 |
| ValueError | input_length不整合 | input_lengthと実際の入力形状が不一致 | 正しいinput_lengthを指定 |
| OOM | メモリ不足 | 大きなinput_dim x output_dimの行列でGPUメモリ不足 | CPU配置に切り替える |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- embedding_lookup_v2はGather操作として最適化
- GPU上では並列的なインデックスアクセスにより高速
- 大きな埋め込み行列はメモリ消費に注意（CPU配置を検討）

## セキュリティ考慮事項

- 埋め込み行列は訓練データの情報を含むため、モデル公開時は注意

## 備考

- 事前学習済み埋め込み（Word2Vec、GloVe、FastText）はembeddings_initializerで設定可能
- `tf.device('cpu:0')`でCPU上に明示配置することでGPU OOMを回避可能
- Flatten -> Dense接続にはinput_lengthの指定が必要

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | embeddings.py | `tensorflow/python/keras/layers/embeddings.py` | Embeddingクラス定義、コンストラクタのパラメータ |

**読解のコツ**: Embeddingレイヤーは内部に`(input_dim, output_dim)`形状の重み行列を1つだけ保持する。`autocast=False`設定がポイントで、入力の浮動小数点キャストを防止している（132行目）。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | embeddings.py | `tensorflow/python/keras/layers/embeddings.py` | Embedding クラス全体（29-213行目） |

**主要処理フロー**:
1. **104-143行目**: __init__ - パラメータ検証（119-122行目: input_dim/output_dim正整数チェック）、初期化器/正則化器/制約の取得
2. **145-154行目**: build - add_weight('embeddings', shape=(input_dim, output_dim))
3. **156-159行目**: compute_mask - mask_zero=Falseなら None、Trueなら math_ops.not_equal(inputs, 0)
4. **161-183行目**: compute_output_shape - input_length考慮の出力形状計算
5. **185-194行目**: call - dtypeチェック/int32キャスト、embedding_lookup_v2、混合精度キャスト
6. **196-212行目**: get_config - シリアライズ用設定辞書の生成

#### Step 3: 低レベル埋め込み検索を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | embedding_ops.py | `tensorflow/python/ops/embedding_ops.py` | embedding_lookup_v2 |

**主要処理フロー**:
- embedding_lookup_v2はGather操作として実装
- パーティション分割された埋め込み行列もサポート

### プログラム呼び出し階層図

```
tf.keras.layers.Embedding(input_dim, output_dim, ...)
    |
    +-- __init__
    |       +-- 入力検証: input_dim > 0, output_dim > 0
    |       +-- initializers.get(embeddings_initializer)
    |       +-- regularizers.get(embeddings_regularizer)
    |       +-- constraints.get(embeddings_constraint)
    |       +-- kwargs['autocast'] = False
    |
    +-- build(input_shape)
    |       +-- self.add_weight('embeddings', (input_dim, output_dim))
    |
    +-- call(inputs)
    |       +-- [dtype != int32/int64] math_ops.cast(inputs, 'int32')
    |       +-- embedding_ops.embedding_lookup_v2(self.embeddings, inputs)
    |       +-- [mixed precision] math_ops.cast(out, compute_dtype)
    |
    +-- compute_mask(inputs, mask)
    |       +-- [mask_zero=True] math_ops.not_equal(inputs, 0)
    |       +-- [mask_zero=False] None
    |
    +-- compute_output_shape(input_shape)
    |       +-- input_shape + (output_dim,)
    |
    +-- get_config()
            +-- {input_dim, output_dim, initializer, regularizer, ...}
```

### データフロー図

```
[入力]                    [処理]                         [出力]

整数テンソル          ---> dtype チェック/キャスト -----> int32テンソル
(batch, seq_len)           |
                           v
埋め込み行列          ---> embedding_lookup_v2 -------> 密ベクトル
(input_dim, output_dim)    |                           (batch, seq_len, output_dim)
                           v
                    [混合精度キャスト]
                           |
                           v
[mask_zero=True]     ---> compute_mask -----------> マスクテンソル
                    not_equal(inputs, 0)           (batch, seq_len)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| embeddings.py | `tensorflow/python/keras/layers/embeddings.py` | ソース | Keras Embeddingレイヤー実装 |
| embedding_ops.py | `tensorflow/python/ops/embedding_ops.py` | ソース | embedding_lookup_v2実装 |
| base_layer.py | `tensorflow/python/keras/engine/base_layer.py` | ソース | Layerベースクラス（add_weight等） |
| base_layer_utils.py | `tensorflow/python/keras/engine/base_layer_utils.py` | ソース | v2_dtype_behavior_enabled |
| backend.py | `tensorflow/python/keras/backend.py` | ソース | floatx()、dtype() |
| initializers.py | `tensorflow/python/keras/initializers.py` | ソース | 初期化器 |
| regularizers.py | `tensorflow/python/keras/regularizers.py` | ソース | 正則化器 |
| constraints.py | `tensorflow/python/keras/constraints.py` | ソース | 制約 |
| math_ops.py | `tensorflow/python/ops/math_ops.py` | ソース | cast、not_equal |
| tf_utils.py | `tensorflow/python/keras/utils/tf_utils.py` | ソース | @shape_type_conversionデコレータ |
